
// DPI Camera
// Sets the current camera frame size to user specified print dimensions
//
// v1.2.2
// 27 Nov 2008
// Matt Gorner
//
// This LScript was written using ConTEXT (http://www.contexteditor.org/)
// Tab width = 3
//
// What does this LScript do?
//
// Calculates camera frame size based on print settings, also includes many common print templates.
// Just add the LScript to LightWave3D Layout and run it, enter your settings or use one of the templates
//
// The script then converts the print dimensions into pixels and sets the camera accordingly.
// Note: This will set your camera's aspect ratio to 1.0, as anything else would be incorrect for print work!
//
// Hope you find it useful!
//
// Cheers
// Matt
//
// 1.2.2
// Added pixel dimensions preview gadget
//

@warnings
@name "DPI Camera"
@script generic

// Declare global gadgets, needed for refreshing gadgets using callback procedures

gad_unit;
gad_width;
gad_height;
gad_bleed;
gad_use_template;
gad_template;
gad_orientation;
gad_preview;

template_gadget_called = FALSE;

// Declare global arrays

template_array;
width_array;
height_array;
measurement_unit_array;
template_gadget_list;

// Declare global document variables

width							= 297; // This variable is updated later on if 'use template' is active
height						= 210; // This variable is updated later on if 'use template' is active
dpi							= 300;
bleed							= 0;
use_template				= TRUE;
template						= 7; // A4 (from template list, see 'template_array[]' variable further down)
measurement_unit 			= 3; // Millimeters, but this variable gets updated later on if 'use template' is active
last_measurement_unit	= 3; // Millimeters, but this variable gets updated later on if 'use template' is active
orientation					= 1; // 1 = Landscape, 2 = Portrait
set_camera					= TRUE;

// LightWave's maximum pixel dimenions for the camera, set here as a variable in case NewTek increase this!
max_camera					= 16000;



// Entry point ...

generic
{

// User interface layout variables

req_width					= 250;
req_height					= 320;

layout_x						= 8;
layout_y						= 15;
layout_width				= 226;
layout_height				= 19;

slot_offset					= 19;
spacer						= 5;
text_offset					= 50;

/*

Fill in templates array:

Format:
template_array[]		= "Width, Height, Unit, String"

Width		= Template width
Height	= Template height
Unit		= Template measurement unit (1 = inches, 2 = centimeters, 3 = millimeters)
String	= Text displayed in the popup gadget

*/

template_array[1]		= "37,26,3,A10 (37 x 26mm)";
template_array[2]		= "52,37,3,A9 (52 x 37mm)";
template_array[3]		= "74,52,3,A8 (74 x 52 mm)";
template_array[4]		= "105,74,3,A7 (105 x 74mm)";
template_array[5]		= "148,105,3,A6 (148 x 105 mm)";
template_array[6]		= "210,148,3,A5 (210 x 148 mm)";
template_array[7]		= "297,210,3,A4 (297 x 210 mm)";
template_array[8]		= "420,297,3,A3 (420 x 297 mm)";
template_array[9]		= "594,420,3,A2 (594 x 420 mm)";
template_array[10]	= "841,594,3,A1 (841 x 594 mm)";
template_array[11]	= "1189,841,3,A0 (1189 x 841 mm)";

// Divider
//(Note: make sure if you add a new divider it has the SAME number of dashes)
template_array[12]	= "0,0,3,-----------------------------------------------";

template_array[13]	= "11, 8.5,1,US Letter (11 x 8.5 in)";
template_array[14]	= "14,8.5,1,US Legal (14 x 8.5 in)";
template_array[15]	= "17,11,1,US Ledger (17 x 11 in)";

// Divider
template_array[16]	= "0,0,3,-----------------------------------------------";

template_array[17]	= "11,8.5,1,ANSI A (Letter 11 x 8.5 in)";
template_array[18]	= "17,11,1,ANSI B (Tabloid 17 x 11 in)";
template_array[19]	= "22,17,1,ANSI C (22 x 17 in)";
template_array[20]	= "34,22,1,ANSI D (34 x 22 in)";
template_array[21]	= "44,34,1,ANSI E (44 x 34 in)";

// Divider
template_array[22]	= "0,0,3,-----------------------------------------------";

template_array[23]	= "12,9,1,Arch A (12 x 9 in)";
template_array[24]	= "18,12,1,Arch B (18 x 12 in)";
template_array[25]	= "24,18,1,Arch C (24 x 18 in)";
template_array[26]	= "36,24,1,Arch D (36 x 24 in)";
template_array[27]	= "48,36,1,Arch E (48 x 36 in)";
template_array[28]	= "42,30,1,Arch E1 (42 x 30 in)";

// Divider
template_array[29]	= "0,0,3,-----------------------------------------------";

template_array[30]	= "5,3,1,Photo (5 x 3 in)";
template_array[31]	= "6,4,1,Photo (6 x 4 in)";
template_array[32]	= "7,5,1,Photo (7 x 5 in)";
template_array[33]	= "10,8,1,Photo (10 x 8 in)";

// Build other arrays from template_array

// loop through template_array

	for( loop = 1; loop <= template_array.size(); loop++ )
	{
		// split each 'template_array' entry (separated by a comma) into a new temp array
		temp = parse(",", template_array[loop] );

		// Build arrays

		width_array					[ loop ] = number( temp[ 1 ] );
		height_array				[ loop ]	= number( temp[ 2 ] );
		measurement_unit_array	[ loop ]	= number( temp[ 3 ] );
		template_gadget_list		[ loop ]	= temp[ 4 ];
	}

	measurement_unit 			= measurement_unit_array[template];
	last_measurement_unit 	= measurement_unit_array[template];

	width 						= width_array	[ template ];
	height 						= height_array	[ template ];

	// Initialise requester

	reqbegin("Print DPI Camera Setting");
	reqsize(req_width, req_height);

	// Layout gadgets

	row_number = 0;

	gad_unit = ctlchoice("Size unit", measurement_unit_array[template] ,@"inches","cm","mm"@);
	ctlposition(gad_unit, layout_x, layout_y + (row_number * slot_offset), layout_width, layout_height, text_offset);

	row_number ++ ;

	gad_width = ctlnumber("Width", width_array[template] );
	ctlposition(gad_width, layout_x, layout_y + (row_number * slot_offset) + spacer, layout_width, layout_height, text_offset);

	row_number ++ ;

	gad_height = ctlnumber("Height", height_array[template] );
	ctlposition(gad_height, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);

	row_number ++ ;

	gad_bleed = ctlnumber("Bleed", bleed);
	ctlposition(gad_bleed, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);

	row_number ++ ;

	gad_dpi = ctlnumber("DPI/PPI", dpi);
	ctlposition(gad_dpi, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);

	row_number ++ ;
	
	gad_orientation = ctlchoice("Format",orientation,@"Landscape","Portrait"@);
	ctlposition(gad_orientation, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);
	
	row_number ++ ;

	gad_separator = ctlsep();
	ctlposition(gad_separator, -2, layout_y + (row_number * slot_offset) + (row_number * spacer) + (layout_height / 2), req_width + 4);

	row_number ++ ;

	gad_use_template = ctlcheckbox("Use Template", use_template);
	ctlposition(gad_use_template, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);

	row_number ++ ;

	gad_template = ctlpopup("Template", template, template_gadget_list );
	ctlposition(gad_template, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);

	row_number ++ ;

	gad_set_camera = ctlcheckbox("Set Camera Frame Size", set_camera);
	ctlposition(gad_set_camera, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);

	row_number ++ ;

	gad_preview = ctlstring("", calculate_camera_size() );
	ctlposition(gad_preview, layout_x, layout_y + (row_number * slot_offset) + (row_number * spacer), layout_width, layout_height, text_offset);
	
	// Define callback procedures for refreshing the gadgets when they are changed ...

	ctlrefresh( gad_unit,			"refresh_unit_gadget"			);
	ctlrefresh( gad_use_template,	"refresh_use_template_gadget" );
	ctlrefresh( gad_template,		"refresh_template_gadget"		);
	ctlrefresh( gad_orientation,	"refresh_orientation_gadget"	);
	ctlrefresh( gad_width,			"refresh_width_gadget"			);
	ctlrefresh( gad_height,			"refresh_height_gadget"			);
	ctlrefresh( gad_bleed,			"refresh_bleed_gadget"			);
	ctlrefresh( gad_dpi,				"refresh_dpi_gadget"				);
	ctlrefresh( gad_set_camera,	"refresh_set_camera_gadget"	);

	// Define callback procedures for gadget ghosting ...

	// When 'gad_use_template' gadget = 1 (on) ghost 'gad_width' and 'gad_height' gadgets (callback returns 0)
	ctlactive(gad_use_template,"toggle_width_height_gadgets_activation", gad_width, gad_height);

	// When 'gad_use_template' gadget = 0 (off) ghost 'gad_template' and 'gad_orientation' gadgets (callback returns 1)
	ctlactive(gad_use_template,"toggle_template_gadgets_activation", gad_template);

	// Show requester, quit if fails

	return if !reqpost();

	// Grab user values
	unit					= getvalue( gad_unit				);
	width					= getvalue( gad_width			);
	height				= getvalue( gad_height			);
	bleed					= getvalue( gad_bleed			);
	use_template		= getvalue( gad_use_template	);
	template				= getvalue( gad_template		);
	orientation			= getvalue( gad_orientation	);
	set_camera			= getvalue( gad_set_camera		);

	// Close requester

	reqend();


	// Calculate and convert into pixels

	bleed_pixels	= bleed	* dpi;
	width_pixels	= width	* dpi;
	height_pixels	= height	* dpi;

	switch( measurement_unit )
	{

	// Inches
	case 1:
		width_pixels 	= ((width_pixels	+ (bleed_pixels * 2) ) );
		height_pixels 	= ((height_pixels + (bleed_pixels * 2) ) );
		feedback_unit = " in ";
		break;

	// Centimeters
	case 2:
		width_pixels 	= ((width_pixels	+ (bleed_pixels * 2) ) / 2.54);
		height_pixels 	= ((height_pixels + (bleed_pixels * 2) ) / 2.54);
		feedback_unit 	= " cm ";
		break;

	// Millimeters
	case 3:
		width_pixels 	= ((width_pixels	+ (bleed_pixels * 2) ) / 25.4);
		height_pixels 	= ((height_pixels + (bleed_pixels * 2) ) / 25.4);
		feedback_unit 	= " mm ";
		break;

	} // switch( measurement_unit )

	// Round pixel value in case of fractional variable
	width_pixels 	= round(width_pixels,0);
	height_pixels 	= round(height_pixels,0);

	// Set camera Frame Size (don't do is width / height is 0 or greater than maximum LW camera frame size)
	if ( set_camera == TRUE & width_pixels > 0 & height_pixels > 0 & width_pixels <= max_camera & height_pixels <= max_camera )
	{
		// Set camera frame size and pixel aspect ratio to 1:0
		FrameSize(width_pixels,height_pixels);
		PixelAspect(1);
	}


	// Test variables to display correct user feedback string ...

	if ( width_pixels > max_camera | height_pixels > max_camera )
	{
		info ("The camera frame size cannot exceed ", max_camera, " pixels, the camera was not modified");
	} else
	{
		if ( set_camera == TRUE )
		{
			if ( bleed > 0)
			{
				info (width, " x ", height, feedback_unit, "+ ", bleed, feedback_unit, "Bleed @ ", dpi, " DPI = Frame Size: ", width_pixels, " x ", height_pixels, " pixels", "<br>Camera Frame Size Adjusted");
			}else
			{
				info (width, " x ", height, feedback_unit, "@ ", dpi, " DPI = Frame Size: ", width_pixels, " x ", height_pixels, " pixels", "<br>Camera Frame Size Adjusted");
			}
			} else
			{
			if ( bleed > 0)
			{
				info (width, " x ", height, feedback_unit, "+ ", bleed, feedback_unit, "Bleed @ ", dpi, " DPI = Frame Size: ", width_pixels, " x ", height_pixels, " pixels", "<br>Camera Frame Size NOT Adjusted");
			}else
			{
				info (width, " x ", height, feedback_unit, "@ ", dpi, " DPI = Frame Size: ", width_pixels, " x ", height_pixels, " pixels", "<br>Camera Frame NOT Size Adjusted");
			}
		} // if (set_camera == TRUE)
	} // max_camera


} // generic


// Gadget refresh callback procedures ...


refresh_unit_gadget: value
{
	// 'Unit gadget' has been changed

	last_measurement_unit	= measurement_unit;
	measurement_unit 			= value;
	convert_settings( last_measurement_unit, measurement_unit );

	// Update width & height gadgets
	setvalue( gad_width, width );
	setvalue( gad_height, height );
	setvalue( gad_bleed, bleed );
}

refresh_use_template_gadget: value
{
	// 'Use template' gadget has been changed, update width / height gadgets to reflect selected template
	
	// Use template = TRUE
	if ( value == TRUE )
	{
		template_gadget_called = TRUE;
	
		if(template_gadget_list[ template ] != "-----------------------------------------------" )
		{
			// 'Template' gadget has been changed, update width / height gadgets to reflect selected template
	
			// Grab width & height of selected template
			width 	= width_array[ template ];
			height 	= height_array[ template ];
		
			// Is the orientation portrait? If so, swap width & height
			if( getvalue( gad_orientation ) == 2 )
			{
				Temp		= width;
				width		= height;
				height	= Temp;
			}

			// Store last measurement unit
			last_measurement_unit	= measurement_unit;
			measurement_unit 			= measurement_unit_array[ template ];
			
			// If user has entered a bleed value, but changes the template, we need to convert the bleed value to the new unit from the selected template
			if ( bleed > 0 & last_measurement_unit != measurement_unit_array[ value ] )
			{
				convert_bleed( last_measurement_unit, measurement_unit );
			}

			// Update width, height and bleed gadgets
			setvalue( gad_width, width );
			setvalue( gad_height, height );
			setvalue( gad_bleed, bleed );

			// Update measurement unit gadget to reflect template default
			setvalue( gad_unit, measurement_unit_array[ template ] );
		}
	}

	// Update 'preview' gadget
	setvalue( gad_preview, calculate_camera_size() );

}

refresh_template_gadget: value
{

	// Don't do anything if user selected a divider (Note: make sure if you add a new divider it has the same number of dashes)

	// Set variable to let the convert_settings function know that this function was called, otherwise it converts twice!
	template_gadget_called = TRUE;

	// Grab which template was selected from gadget value
	template = value;

	// Ignore if a divider was selected
	if(template_gadget_list[ value ] != "-----------------------------------------------" )
	{
		// 'Template' gadget has been changed, update width / height gadgets to reflect selected template

		// Grab width & height of selected template
		width 	= width_array[ value ];
		height 	= height_array[ value ];
	
		// Is the orientation portrait? If so, swap width & height
		if( getvalue( gad_orientation ) == 2 )
		{
			temp		= width;
			width		= height;
			height	= temp;
		}

		// Store last measurement unit
		last_measurement_unit	= measurement_unit;
		measurement_unit 			= measurement_unit_array[ value ];

		// If user has entered a bleed value, but changes the template, we need to convert the bleed value to the new unit from the selected template
		if ( bleed > 0 & last_measurement_unit != measurement_unit_array[ value ] )
		{
			convert_bleed( last_measurement_unit, measurement_unit );
		}

		// Update width, height and bleed gadgets
		setvalue( gad_width, width );
		setvalue( gad_height, height );
		setvalue( gad_bleed, bleed );

		// Update measurement unit gadget to reflect template default
		setvalue( gad_unit, measurement_unit_array[ value ] );
	}
	
}

refresh_orientation_gadget: value
{
	// 'Orientation' gadget has been changed, swap width and height values each time this gadget is called

	temp		= width;
	width		= height;
	height	= temp;
	
	// Update width and height gadgets
	setvalue( gad_width, width );
	setvalue( gad_height, height );
}

refresh_width_gadget: value
{
	// Update global width variable after gadget has changed
	width = value;

	// Update 'preview' gadget
	setvalue( gad_preview, calculate_camera_size() );
}

refresh_height_gadget: value
{
	// Update global height variable after gadget has changed
	height = value;

	// Update 'preview' gadget
	setvalue( gad_preview, calculate_camera_size() );
}

refresh_bleed_gadget: value
{
	// Update global bleed variable after gadget has changed
	bleed = value;

	// Update 'preview' gadget
	setvalue( gad_preview, calculate_camera_size() );
}

refresh_dpi_gadget: value
{
	// Update global dpi variable after gadget has changed
	dpi = value;
	
	// Update 'preview' gadget
	setvalue( gad_preview, calculate_camera_size() );
}

refresh_set_camera_gadget: value
{
	// Update global set camera variable after gadget has changed
	set_camera = value;
}


// Gadget visibility callback procedures ...


toggle_width_height_gadgets_activation: value
{
	return(value == 0);
}

toggle_template_gadgets_activation: value
{
	return(value == 1);
}


// Custom procedures ...


convert_settings: old_unit, new_unit
{

// Don't do if template gadget was just refreshed by another gadget
if ( template_gadget_called == FALSE )
{

	// Convert current dimensions

	switch( new_unit )
	{
 			// New Unit = Inches
			case 1:

				// Was cm before
				if( old_unit == 2 )
				{
					// info("Converting from centimeters to inches");
					width		= width 	* 0.393700787402;
					height	= height	* 0.393700787402;
					bleed		= bleed	* 0.393700787402;
				} else
				{
					// info("Converting from millimeters to inches");
					width		= width 	* 0.03937007874;
					height	= height	* 0.03937007874;
					bleed		= bleed	* 0.03937007874;
				}
				break;

			// New Unit = Centimeters
			case 2:

				// Was inches before
				if( old_unit == 1 )
				{
					// info("Converting from inches to centimeters");
					width		= width 	* 2.54;
					height	= height	* 2.54;
					bleed		= bleed	* 2.54;
				}else
				{
					// info("Converting from millimeters to centimeters");
					width		= width 	* 0.1;
					height	= height	* 0.1;
					bleed		= bleed	* 0.1;
				}
				break;

			// New Unit = Millimeters
			case 3:

				// Was inches before
				if( old_unit == 1 )
				{
					// info("Converting from inches to millimeters");
					width		= width 	* 25.4;
					height	= height	* 25.4;
					bleed		= bleed	* 25.4;
				}else
				{
					// Was cm before
					// info("Converting from centimeters to millimeters");
					width		= width 	* 10;
					height	= height	* 10;
					bleed		= bleed	* 10;
				}
				break;

	} // switch( new_unit )

} else
{
	// Template Gadget was called, so reset the variable that said it was, otherwise this test won't work next time it is!
	template_gadget_called = FALSE;
}

} // convert_settings

convert_bleed: old_unit, new_unit
{
	// Convert current dimensions

	switch( new_unit )
	{
 			// New Unit = Inches
			case 1:

				// Was cm before
				if( old_unit == 2 )
				{
					// info("Converting from centimeters to inches");
					bleed		= bleed	* 0.393700787402;
				} else
				{
					// info("Converting from millimeters to inches");
					bleed		= bleed	* 0.03937007874;
				}
				break;

			// New Unit = Centimeters
			case 2:

				// Was inches before
				if( old_unit == 1 )
				{
					// info("Converting from inches to centimeters");
					bleed		= bleed	* 2.54;
				}else
				{
					// info("Converting from millimeters to centimeters");
					bleed		= bleed	* 0.1;
				}
				break;

			// New Unit = Millimeters
			case 3:

				// Was inches before
				if( old_unit == 1 )
				{
					// info("Converting from inches to millimeters");
					bleed		= bleed	* 25.4;
				}else
				{
					// Was cm before
					// info("Converting from centimeters to millimeters");
					bleed		= bleed	* 10;
				}
				break;

	} // switch( measurement_unit )

} // convert_settings


calculate_camera_size
{
	// Calculate and convert into pixels
 	bleed_pixels	= bleed	* dpi;
	width_pixels	= width	* dpi;
	height_pixels	= height	* dpi;

	// Convert from whichever measurement unit was selected
	switch( measurement_unit )
	{

	// Inches
	case 1:
		width_pixels 	= ((width_pixels	+ (bleed_pixels * 2) ) );
		height_pixels 	= ((height_pixels + (bleed_pixels * 2) ) );
		break;

	// Centimeters
	case 2:
		width_pixels 	= ((width_pixels	+ (bleed_pixels * 2) ) / 2.54);
		height_pixels 	= ((height_pixels + (bleed_pixels * 2) ) / 2.54);
		break;

	// Millimeters
	case 3:
		width_pixels 	= ((width_pixels	+ (bleed_pixels * 2) ) / 25.4);
		height_pixels 	= ((height_pixels + (bleed_pixels * 2) ) / 25.4);
		break;

	} // switch( measurement_unit )

	// Round pixel value in case of fractional values
	width_pixels 	= round(width_pixels,0);
	height_pixels 	= round(height_pixels,0);
	
	// Build 'preview' string
	str = string(width_pixels) + " x " + string(height_pixels) + " pixels";
	
	// Return back the string to whatever called this function
	return(str);
}

